#ifndef PORT_MESSAGE_H
#define PORT_MESSAGE_H

// Class for sending/receiving messages through ports to Experimenter or Rat
// from the environment or experimenter respectively.

/*
// Changes
// CGP, 11/2/00: Previously all port messages were just simple message types.
//		The port messages did not contain any additional data. Now there is a need
//		for these messages to contain additional data. This is because I want
//		the messages to just be single messages, and when the experimenter
//		is informed of the rat moving or the rat consuming food, via it's port,
//		it needs to know two things: (a) the fact that the rat moved/consumed food,
//		and (b) the arm number where the movement/food consuming took place.
//		This interface will now be changed so that full BMessages are sent to the
//		rat and experimenter via the port. Right now, only EXPMSG_RAT_MOVED and
//		EXPMSG_RAT_CONSUMED_FOOD messages will contain information in addition
//		to the message type.
*/

#include <OS.h>
#include <List.h>
#include "Message.h"

// CGP, 1/9/01, THESE MESSAGE TYPES SHOULD BE PUT IN A DIFFERENT PLACE! Specific to
// experimenters and rats. More parts of the system are or will be using PortMessage.h/.cpp
// than the rat & experimenter.

// Message types
enum {
	// from experimenter to rat
	RATMSG_GO = 0, // let rat run around from where she is (run mode only)
	RATMSG_STOP, // force rat to stop where he is
	RATMSG_RESET, // rat resets his or her brain state to the start state
	RATMSG_STEP_MODE, // put rat in single step mode
	RATMSG_RUN_MODE, // put rat in run mode (continuous running)
	RATMSG_SINGLE_STEP, // run a single step when rat is in step mode
	
	// messages to experimenter from environment. Also contain arm number.
	EXPMSG_RAT_MOVED, // let experimenter know rat moved (from Environment)
	EXPMSG_RAT_CONSUMED_FOOD, // let experimenter know rat at food at arm (from Environment)
	
	// From GUI to experimenter
	EXPMSG_RUN_TRIAL, // From GUI
	//EXPMSG_START_STEP, // From GUI
	EXPMSG_NEXT_STEP, // From GUI
	EXPMSG_COMPLETE_CURRENT_RAT,
	EXPMSG_SETUP // special: from ExperimenterPlugin.cpp; see documentation in Start there.
};

class PortMessage : public Message {
	public:
		// If buffered is true then the port is created with a buffering
		// scheme that enables an unbounded (based on available memory)
		// number of messages to be pending (unreceived). If buffered is
		// false (default) then only a small (system dependent) number
		// of messages can be pending.
		PortMessage(char *name=NULL); // create a port
		~PortMessage();
		bool IsMessageReady(); // is a message ready to be received?
		
		// Peek at a message that is ready. This does not remove the
		// message from the incoming messages. GetMessage has to be used
		// to remove. Blocks if no message ready. 
		BMessage *PeekMessage();
		
		// Retrieve message (blocks if none ready)
		// Caller has ownership of mesage. All messages are received as BMessage's
		BMessage *GetMessage();
		
		// remove any pending messages from the incoming message queue
		void Flush();
		
		// Messages can be sent as codes (e.g., RATMSG_GO) or BMessages.
		// If a message is sent as a code, then the code will be in the
		// the "which" field of the message in GetMessage. Such a code message
		// will contain no other information.
		// SendMessage will cause the caller to block if ther queue of messages
		// in the port happens to be full. The caller will unblock when the
		// receiver of the messages does a PeekMessage or GetMessage or Flush.
		void SendMessage(BMessage *msg); // caller retains ownership of message
		void SendMessage(int msg_code);
		
	private:
		port_id port; // BeOS port
		BMessage *peekBuffer; // single buffer for message peeked at
};

#endif